/**
 * ActiveBackBuffer.java
 * v1.0b
 * 
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 * 
 *** LOG ***
 * 27.Dec.1999.
 * ~~~~~~~~~~~~
 *	18:49..	- created
 */

public class ActiveBackBuffer
	extends BackBuffer
{
	int m_iObjectCount = 0;				// number of objects in array
	int m_iObjectArraySize = 100;		// actual size of array
	ActiveBackBuffer m_ObjectArray[];	// array of BackBuffer child objects

	/**
	 * m_lTimeUntilNextAction is time that must pass until object will do its next action.
	 * It may be:
	 * -2 if object needs time as fast as he can..
	 * -1 if object has no particular action
	 *  0 if object must do action right now, without spending any time
	 * >0 if object has to wait specified amount of time for next action
	 */
	private long m_lTimeUntilNextAction = -1;		// in milliseconds
	
	
	/** CONSTRUCTION **********************************************************/

	public ActiveBackBuffer()
	{
		m_ObjectArray = new ActiveBackBuffer[m_iObjectArraySize + 1];
	}


	/**
	 * Spends given time (in milliseconds), and triggers action(s) when counts 
	 * m_lTimeTillNext action to zero.
	 */
	public void action(long lTimeForAction)
	{
		long lMinTime;
		
		while(true)
		{
			// Find smallest time for action...
			lMinTime = getMinTimeUntilNextAction(lTimeForAction);
			
			// If there are no objects waiting for action - just return
			if (lMinTime == -1)
				break;

			// Let all objects do their action...
			checkToDoNextAction(lMinTime);
			lTimeForAction -= lMinTime;
			
			if (lTimeForAction <= 0)
				break;
		}
	}
	
	
	/**
	 * Queries this object to execute its next action if specified
	 * time is time of its next action. (and to parse this call to
	 * all its childs).
	 * Specified time was calculated before and represents smallest 
	 * nextActionTime of all objects.
	 * @param lExpectedActionTime
	 */
	public void checkToDoNextAction(long lExpectedActionTime)
	{
		long lTimeForAction = lExpectedActionTime;
		
		// do action of this object
		while(true)
		{
			if (m_lTimeUntilNextAction == -2)	// continuos?
			{
				lTimeForAction = doContinuousAction(lTimeForAction);
			}
			else if (m_lTimeUntilNextAction == -1)	// no action?
			{
				// ignore
				break;
			}
			else if (m_lTimeUntilNextAction == 0)	// has next action?
			{
				doNextAction();
			}
			else if (m_lTimeUntilNextAction == lTimeForAction)
			{
				m_lTimeUntilNextAction = -1;
				lTimeForAction = 0;
				doNextAction();
			}
			else if (m_lTimeUntilNextAction > lTimeForAction)
			{
				m_lTimeUntilNextAction -= lTimeForAction;
				lTimeForAction = 0;
			}
			else if (m_lTimeUntilNextAction < lTimeForAction)
			{
				lTimeForAction -= m_lTimeUntilNextAction;
				m_lTimeUntilNextAction = 0;
				doNextAction();
			}
			// m_lTimeUntilNextAction cannot be less than lExpectedActionTime
			// (because lExpectedAction time is smallest)
			
			if (lTimeForAction <= 0)
				break;		
		}
		
		
		// and check for its child objects
		for (int i=0; i < m_iObjectCount; i++)
		{
			m_ObjectArray[i].checkToDoNextAction(lExpectedActionTime);
		}
	}
	

	
	public void doNextAction()
	{
		// this will be overriden by child class of this object
		
		m_lTimeUntilNextAction = -1;
	}
	
	/**
	 * Does continuous action.
	 * @return time left after action
	 */
	public long doContinuousAction(long lTimeForAction)
	{
		// This will be overriden..
		// But to be sure - return 0 time left (spend all given time)
		return 0;
	}

	
	/**
	 * Returns smallest period until next action of this object (or
	 * any of its childs)
	 */
	public long getMinTimeUntilNextAction(long lCurrentMinTime)
	{
		// while this object needs to do action right now
		while (m_lTimeUntilNextAction == 0)
		{
			doNextAction();
		}
			
		if (m_lTimeUntilNextAction == -2)		// continous?
		{
			// lCurrentMinTime = lCurrentMinTime
		}
		else if (m_lTimeUntilNextAction == -1)	// no action
		{
			// ignore lCurrentMin			
		}
		else if (m_lTimeUntilNextAction > 0)
		{
			if (m_lTimeUntilNextAction < lCurrentMinTime)
			{
				lCurrentMinTime = m_lTimeUntilNextAction;
			}
		}
		
		// parse this call to child objects..
		for (int i=0; i < m_iObjectCount; i++)
		{
			lCurrentMinTime = m_ObjectArray[i].getMinTimeUntilNextAction(lCurrentMinTime);
		}
		
		return lCurrentMinTime;
	}
	
	
	public void setActionWhenTime(long lTimeOfNextAction)
	{
		m_lTimeUntilNextAction = lTimeOfNextAction;
	}
	
	public void setActionContinuous()
	{
		m_lTimeUntilNextAction = -2;
	}
	
	public void setActionNone()
	{
		m_lTimeUntilNextAction = -1;
	}
	
	public void setActionNext()
	{
		m_lTimeUntilNextAction = 0;
	}

	/** OBJECT MANIPULATION ************************************************/	
	
	/**
	 * Call this to add your child BackBuffer to this BackBuffer
	 */
	public void addObject(ActiveBackBuffer o)
	{
		if (m_iObjectCount < m_iObjectArraySize)
		{
			m_ObjectArray[m_iObjectCount] = o;
			m_iObjectCount++;
			o.m_bChanged = true;
		}
		
		super.addObject(o);
	}
	
	/**
	 * Removes given object from backbuffer...
	 * Call 'paint' after this to refresh backBuffer!
	 */
	public void removeObject(ActiveBackBuffer o)
	{
		// find object in array...
		for (int i=0; i < m_iObjectCount; i++)
		{
			if (m_ObjectArray[i] == o)
			{
				m_RectangleManager.addRectangle(o.x, o.y, o.width, o.height);
				
				m_ObjectArray[i] = null;
				
				// remove this object - shift all after him one place left..
				while(i < m_iObjectCount)
				{
					m_ObjectArray[i] = m_ObjectArray[i+1];
					m_ObjectArray[i+1] = null;
					i++;
				}
				
				m_iObjectCount--;
				break;
			}
		}
		
		super.removeObject(o);
	}

		
	
}
