/**
 * BackBuffer.java
 * v1.0b
 * 
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 * 
 * BackBuffer class represents BackBuffer painting engine.
 * 
 * BackBuffer can be used as:
 *	1. Parent
 *	2. Child
 *  3. SubParent
 * 
 * 1) PARENT
 *		BackBuffer as Parent is image with some 'width' and 'height',
 *		and objects on it.. You will use addObject method to add
 *		Child objects and SubParents to this BackBuffer, also,
 *		you will use
 *			paint(yourDestinationGraphics, yourDestX, yourDestY)
 *		or
 *			update(yourDestinationGraphics, yourDestX, yourDestY)
 *		to paint or update content of this Parent on e.g. your Applet, 
 *		or some memory image.
 *		You won't add this BackBuffer to some other BackBuffer - because
 *		if you do, then this BackBuffer may be considered as SubParent
 *		(check description of SubParent below).
 * 
 *		Example:
 * 
 *			BackBuffer parent = new BackBuffer();
 *			parent.setImage(yourBackgroundImg);	// Set its background image
 *			parent.addObject(someChild1);		// Add 1st child BackBuffer
 *			parent.addObject(someChild2);		// Add 2nd child BackBuffer
 *			parent.addObject(someSubParent);	// Add some SubParent BackBuffer (already filled with Childs)
 *			:
 *			parent.paint(yourDestinationGraphics, yourDestX, yourDestY);	// paint all..
 *			:
 *			parent.update(yourDestinationGraphics, yourDestX, yourDestY);	// update what has changed..
 *			:
 * 
 * 2) CHILD
 *		BackBuffer as Child is image with some 'width' and 'height'
 *		but without any objects on it. This BackBuffer has only its
 *		image - you can consider it as 'single object'. So, you will 
 *		set its image, and will add it to some Parent or SubParent 
 *		BackBuffer.
 *		You usually won't use any painting methods of this Child object, 
 *		because, this Child will be painted on his Parent or SubParent 
 *		and they will manage painting and refreshing it on screen.
 * 
 *		Example:
 * 
 *			BackBuffer child = new BackBuffer();	// Create child BackBuffer
 *			child.setImage(yourSpriteImage);		// Set its image
 *			parentBackBuffer.addObject(child);		// Add it to some Parent or SubParent BackBuffer 
 *													// (check above and below)
 *			parentBackBuffer.paint(yourDestinationGraphics, yourDestX, yourDestY);	// paint new object
 *			:
 *			child.moveObject(iNewX, iNewY);			// Change position
 *			:
 *			parentBackBuffer.update(yourDestinationGraphics, yourDestX, yourDestY);	// update object movement
 * 
 * 3) SUBPARENT
 *		BackBuffer as SubParent is same as 'Parent', but in contrast - has
 *		its own parent. So, it is added to some real Parent or another 
 *		SubParent (adding it to SubParent, it will become SubSubParent).
 *		Like with Child, you also won't use	its painting 
 *		methods - because Parent will manage its refresh.
 * 
 *		Example:
 * 
 *			BackBuffer subParent = new BackBuffer();
 *			subParent.setImage(someBackgroundImg);	// Set its background image
 *			subParent.addObject(someChild1);		// Add 1st child BackBuffer to this subParent
 *			subParent.addObject(someChild2);		// Add 2nd child BackBuffer to this subParent
 *			subParent.addObject(someSubSubParent);	// Add some SubParent BackBuffer (already filled with Childs) 
 *													// to this (this one will be its parent)
 *			realParent.addObject(subParent);		// Add this SubParent to its parent (or to some other SubParent)
 *			realParent.paint(yourDestinationGraphics, yourDestX, yourDestY);	// paint new object (SubParent)
 *			:
 *			someChild.moveObject(x, y);				// Move object on subParent
 *			realParent.update(yourDestinationGraphics, yourDestX, yourDestY);	// update object movement
 *			:
 *			subParent.moveObject(iNewX, iNewY);		// Move entire subParent on realParent
 *			realParent.update(yourDestinationGraphics, yourDestX, yourDestY);	// update subParent movement
 *			:
 * 
 * CREATION RULE:
 * 
 * After creating BackBuffer, you must use
 *  
 *		setImage(someBackgroundImage)
 * 
 * to set backround image (and initialize BackBuffer) or use 
 * 
 *		create(width, height, color, component, imob) 
 * 
 * to create backbuffer with specified width and height, 
 * but with specified background color instead of background image.
 *		
 *** TODO IN FUTURE ***
 * 
 * Possible optimization: when painting some non-transparent child backbuffer,
 * there is no need to paint background of parent on that position...
 * 
 *** LOG ***
 * 12.Dec.1999.
 * ~~~~~~~~~~~~
 *	- created basic sceleton
 *
 * 19.Dec.1999.
 * ~~~~~~~~~~~~
 *  - Buffer in Buffer works!
 *
 * 20.Dec.1999.
 * ~~~~~~~~~~~~
 *  - Added showObject() and hideObject()
 *
 */
import java.applet.*;
import java.awt.*;
import java.awt.image.*;

public class BackBuffer
	extends RectangleManagerRect
{
	/**
	 * Old position (when object has moved)
	 */
	int m_iOldX;
	int m_iOldY;
	boolean m_bMoved = false;		// true when object was moved since last update

	/**
	 * Old size (when object has resized)
	 */
	int m_iOldWidth;
	int m_iOldHeight;
	boolean m_bResized = false;		// true when object was resize since last update

	/**
	 * Actual (BackBuffer) image of this object
	 */
	protected Image m_image;

	/**
	 * PROPERTIES
	 */
	
	boolean m_bVisible		= true;

	boolean m_bNoWidthAndHeight = false;

	String m_strName = new String();
								  
	/**
	 * BACKGROUND
	 * 
	 * When updating some rectangle (in m_image), first, background 
	 * is painted, and then all other child objects).
	 * Background can be:
	 *	- image (when m_imgBackground != null)
	 *  - color (when m_imgBackground == null and m_colorBackground != null)
	 *  - none (when m_imgBackground == null and m_colorBackground == null)
	 */
	protected Image m_imgBackground;	// Reference to given background image for BackBuffer
											// null if not to be used
	protected Color m_colorBackground;	// Color object if color should be used as background
											// null if not to be used

	boolean m_bChanged = true;			// true when background image has changed

	// ...

	protected ImageObserver	m_imob;		// given ImageObserver to be used for painting
	protected Component m_component;	// given Component from parent

	protected RectangleManager m_RectangleManager;	// Handles rectangles (optimizations for repainting)

	boolean m_bDebug = false;		// true when in debug mode (painting borders of painted rectangles)

	/**
	 * BACKBUFFER OBJECTS
	 *
	 * BackBuffer objects (Childs and SubParents) are stored in dynamic array.
	 * User adds its objects by AddObject method to this BackBuffer.
	 * First object in array (index 0) is first painted - and is actually bottom
	 * object
	 * Last object in array (index m_iObjectCount-1) is last one to be painted
	 * and is actually top-most object (above all objects).
	 */

	int m_iObjectCount = 0;				// number of objects in array
	int m_iObjectArraySize = 100;		// actual size of array
	BackBuffer m_ObjectArray[];			// array of BackBuffer child objects

	/** CONSTRUCTION **********************************************************/

	public BackBuffer()
	{
		// m_iX = 0;
		// m_iY = 0;

		m_ObjectArray = new BackBuffer[m_iObjectArraySize + 1];
		m_RectangleManager = new RectangleManager();

		// must be repainted first..
		m_bChanged = true;
	}

	/** CREATION METHODS ******************************************************/

	/**
	 * If you don't use "setImage(bkImage)" after constructing BackBuffer,
	 * you must call this one to initialize backBuffer to wanted width,
	 * height and to use specified background color instead of background
	 * image.
	 */
	public void create(int iWidth, int iHeight,
					   Color colorBackground, Component c, ImageObserver imob)
	{
		// remember ImageObserver
		m_imob = imob;

		// we'll use background color instead of background picture
		m_imgBackground = null;
		m_colorBackground = colorBackground;

		// get width and height for BackBuffer
		width = iWidth;
		height = iHeight;

		// create BackBuffer image
		m_image = c.createImage(width, height);

		// must be repainted..
		m_bChanged = true;
	}


	/**
	 * Use this to create this backbuffer as transparent one.
	 */
	public void createTransparent(int iWidth, int iHeight, Component c, ImageObserver imob)
	{
		// remember ImageObserver
		m_imob = imob;

		// we'll use background color instead of background picture
		m_imgBackground = null;
		m_colorBackground = null;

		// get width and height for BackBuffer
		width = iWidth;
		height = iHeight;

		// must be repainted..
		m_bChanged = true;
	}


	/**
	 * Use this to create this backbuffer that will be ready to receive image
	 */
	public void createForImage(Component c, ImageObserver imob)
	{
		// remember ImageObserver
		m_imob = imob;
		m_component = c;

		// we'll use background color instead of background picture
		m_imgBackground = null;
		m_colorBackground = null;
	}


	/** REPAINT METHODS *******************************************************/

	/**
	 * Updates destination graphics only with updated BackBuffer contents.
	 * iDestX and iDestY are coordinates of this BackBuffer on destination.
	 */
	public void update(Graphics gDest, int iDestX, int iDestY)
	{
		//
		// Find and update rectangles changed (in BackBuffer,
		// rectangles of invalidated objects)
		//
		updateAndGetInvalidation(m_RectangleManager);

		//
		// Paint updated rectangles to destination
		//

		RectangleManagerRect r = m_RectangleManager.m_RectangleList;
		while(r != null)
		{
			// copy it from backbuffer to screen..
			gDest.drawImage(m_image,
						r.x + iDestX, r.y + iDestY, iDestX + r.x + r.width, iDestY + r.y + r.height,	// dest
						r.x, r.y, r.x + r.width, r.y + r.height,	// source
						Color.white,
						m_imob);

			if (m_bDebug)
				gDest.drawRect(r.x + iDestX, r.y + iDestY, r.width - 1, r.height - 1);
			
			r = r.m_next;
		}
		m_RectangleManager.deleteAll();
	}
	

	/**
	 * Paints entire backbuffer to destination
	 */
	public void paint(Graphics gDest, int iDestX, int iDestY)
	{
		updateAndGetInvalidation(m_RectangleManager);
		
		// And paint whole BackBuffer to destination at specified coordinates
		gDest.drawImage(m_image, iDestX + x, iDestY + y, m_imob);
	}
	
	
	
	/**
	 * This will repaint all invalidated areas inside this object,
	 * and add repainted rectangles as invalidation rectangles to
	 * parent (in coordinates relative to parent).
	 * 
	 * @param int iParentX			- parent X coordinate on destination
	 * @param int iParentY			- parent Y coordinate on destination
	 * @param RectangleManager rm	- parent's rectangle manager to
	 *		fill with invalidation rectangles (if null, do not fill)
	 */
	public void updateAndGetInvalidation(RectangleManager rm)
	{
		//
		// Find rectangles to repaint in BackBuffer
		// (collect and optimize all rectangles of invalidated objects)
		//
		
		findRectanglesToRepaint(m_RectangleManager);
		m_RectangleManager.optimizeForRepaint();
		
		//
		// Repaint all rectangles
		//
		
		RectangleManagerRect r = m_RectangleManager.m_RectangleList;
		
		while(r != null)
		{
			if (m_image != null)
			{
				Graphics g = m_image.getGraphics();
				paintRect(g, r);
			}
			
			// if requested - add this repainted rectangle as invalidationArea for parent
			if (rm != null)
				rm.addRectangle(x + r.x, y + r.y, r.width, r.height);
			
			r = r.m_next;
		}
		if (rm != m_RectangleManager)
			m_RectangleManager.deleteAll();
	}

	
	/**
	 * Scans all objects, updates SubParents, finds rectangles needed 
	 * to be repainted on screen, and adds them as invalidated areas 
	 * to given RectangleManager.
	 * Also, marks all objects as repainted (not changed, not moved,
	 * not resized).
	 * Given RectangleManager will be filled with rectangles with
	 * coordinates relative to this BackBuffer.
	 */
	protected void findRectanglesToRepaint(RectangleManager rm)
	{
		RectangleManagerRect rBackBuffer = new RectangleManagerRect();
		rBackBuffer.setPosAndSize(0, 0, width, height);
		
		BackBuffer o;

		RectangleManagerRect rObject = new RectangleManagerRect();
		
		// browse through all objects..
		for (int i = 0; i < m_iObjectCount; i++)
		{
			o = m_ObjectArray[i];
			
			if (o.m_bMoved && o.m_bResized)
			{
				// check to add old object position&size
				rObject.setPosAndSize(o.m_iOldX, o.m_iOldY, o.m_iOldWidth, o.m_iOldHeight);
				if (rObject.boundToRect(rBackBuffer))
				{
					rm.addRectangle(rObject.x, rObject.y, rObject.width, rObject.height);
				}
			}
			else if (o.m_bMoved && !o.m_bResized)
			{
				// check to add old object position&size
				rObject.setPosAndSize(o.m_iOldX, o.m_iOldY, o.width, o.height);
				if (rObject.boundToRect(rBackBuffer))
				{
					rm.addRectangle(rObject.x, rObject.y, rObject.width, rObject.height);
				}
			}
			else if (!o.m_bMoved && o.m_bResized)
			{
				// check to add old object position&size
				rObject.setPosAndSize(o.x, o.y, o.m_iOldWidth, o.m_iOldHeight);
				if (rObject.boundToRect(rBackBuffer))
				{
					rm.addRectangle(rObject.x, rObject.y, rObject.width, rObject.height);
				}
			}
			
			// if object is inside this BackBuffer area..
			rObject.setPosAndSize(o.x, o.y, o.width, o.height);
			if (rObject.boundToRect(rBackBuffer))
			{
				if (o.m_bMoved || o.m_bResized)
				{
					// add his entire (new) rectangle to repaint list
					rm.addRectangle(rObject.x, rObject.y, rObject.width, rObject.height);
				}
				
				o.m_bMoved = false;
				
				// check to update object, and remember if any new invalidations..
				o.updateAndGetInvalidation(rm);				
			}
			
			o.m_bMoved = false;
			o.m_bResized = false;
			o.m_bChanged = false;
		}

		// if this object has changed or resized...
		if (m_bChanged || m_bResized)
		{
			// then add whole this object
			rm.addRectangle(0, 0, width, height);
			m_bChanged = false;
			m_bResized = false;
		}
	}
	
	
	public int getObjectCount()
	{
		return m_iObjectCount;
	}

	
	/**
	 * Call this to add your child BackBuffer to this BackBuffer
	 */
	public void addObject(BackBuffer o)
	{
		if (m_iObjectCount < m_iObjectArraySize)
		{
			m_ObjectArray[m_iObjectCount] = o;
			m_iObjectCount++;
			o.m_bChanged = true;
		}
	}
	
	/**
	 * Removes given object from backbuffer...
	 * Call 'paint' after this to refresh backBuffer!
	 */
	public void removeObject(BackBuffer o)
	{
		// find object in array...
		for (int i=0; i < m_iObjectCount; i++)
		{
			if (m_ObjectArray[i] == o)
			{
				// add his current rectangle
				m_RectangleManager.addRectangle(o.x, o.y, o.width, o.height);
				
				// and old-position rectangle
				if ((o.m_bMoved) && (!o.m_bResized))
					m_RectangleManager.addRectangle(o.m_iOldX, o.m_iOldY, o.width, o.height);
				else if ((o.m_bResized) && (!o.m_bMoved))
					m_RectangleManager.addRectangle(o.x, o.y, o.m_iOldWidth, o.m_iOldHeight);
				else if ((o.m_bResized) && (o.m_bMoved))
					m_RectangleManager.addRectangle(o.m_iOldX, o.m_iOldY, o.m_iOldWidth, o.m_iOldHeight);
				
				m_ObjectArray[i] = null;
				
				// remove this object - shift all after him one place left..
				while(i < m_iObjectCount)
				{
					m_ObjectArray[i] = m_ObjectArray[i+1];
					m_ObjectArray[i+1] = null;
					i++;
				}
				
				m_iObjectCount--;
				return;
			}
		}
	}

	public void moveObjectToBack(BackBuffer o)
	{
		// find object in array...
		for (int i=0; i < m_iObjectCount; i++)
		{
			if (m_ObjectArray[i] == o)
			{
				m_RectangleManager.addRectangle(o.x, o.y, o.width, o.height);
				
				// remove this object - shift all before it one place right..
				while(i > 0)
				{
					m_ObjectArray[i] = m_ObjectArray[i-1];
					i--;
				}
				
				// and put object at top of list (Back)
				m_ObjectArray[i] = o;
				return;
			}
		}
	}
	
	/**
	 * Hides object
	 */
	public void hideObject(BackBuffer o)
	{
		// find object in array...
		for (int i=0; i < m_iObjectCount; i++)
		{
			if (m_ObjectArray[i] == o)
			{
				if (o.m_bVisible)
				{
					// before hiding it - add it to invalidation list...
					m_RectangleManager.addRectangle(o.x, o.y, o.width, o.height);
					o.m_bVisible = false;
				}
				return;
			}
		}
	}
	
	/**
	 * Hides object
	 */
	public void showObject(BackBuffer o)
	{
		// find object in array...
		for (int i=0; i < m_iObjectCount; i++)
		{
			if (m_ObjectArray[i] == o)
			{
				if (!o.m_bVisible)
				{
					// before hiding it - add it to invalidation list...
					m_RectangleManager.addRectangle(o.x, o.y, o.width, o.height);
					o.m_bVisible = true;
				}
				return;
			}
		}
	}
	
	/** THIS OBJECT MANIPULATION ********************************************/
	
	/**
	 * Sets new (background) image for this object
	 */
	public void setImage(Image imgBackground)
	{
		// if same image - ignore
		if (imgBackground == m_imgBackground)
			return;
		
		// remember old size..
		m_iOldWidth = width;
		m_iOldHeight = height;
		
		// set new background image
		m_imgBackground = imgBackground;
		
		// get new width and height for this backbuffer
		width = m_imgBackground.getWidth(m_imob);
		height = m_imgBackground.getHeight(m_imob);

		// if new size
		if ((m_iOldWidth != width) || (m_iOldHeight != height))
		{
			// create new BackBuffer image (because of new size)
			m_image = m_component.createImage(width, height);
		}
		
		// if there was no image...
		if (m_imgBackground == null)
		{
			// mark that this is first size, so there was no resize..
			m_iOldWidth = width;
			m_iOldHeight = height;
		}

		// if different size - mark resized
		if ((m_iOldWidth != width) || (m_iOldHeight != height))
			m_bResized = true;
		
		// mark changed
		m_bChanged = true;
	}
	
	/**
	 * Moves object to new position.
	 */
	public void moveObject(int iNewX, int iNewY)
	{
		// if same position - ignore
		if ((x == iNewX) && (y == iNewY))
			return;
						   
		// if object was not yet moved from this position
		if (!m_bMoved)
		{
			m_iOldX = x;
			m_iOldY = y;
			m_bMoved = true;
		}
		x = iNewX;
		y = iNewY;
	}

	/**
	 * Marks object as painted (refreshed).
	 * (all 'changed' flags are cleared)
	 */
	public void setPainted()
	{
		m_bChanged = false;
		m_bMoved = false;
		m_bResized = false;
		
		m_iOldX = x;
		m_iOldY = y;
		m_iOldWidth = width;
		m_iOldHeight = height;
	}

	/**
	 * Repaints rectangle in backBuffer of this object
	 * @param RectangleManagerRect rArea - rectangle to be repainted
	 *		in relative coordinates to this object
	 */
	protected void paintRect(Graphics g, RectangleManagerRect rArea)
	{
		// First, try to bound specified rectangle to this BackBuffer
		Rectangle r = new Rectangle(0, 0, width, height);
		if (rArea.boundToRect(r))
		{
			// if not transparent
			if (m_image != null)
			{
				//Graphics g = m_image.getGraphics();
				
				// Then paint background to BackBuffer (only specified rectangle)
				paintBackgroundRect(rArea);
		
				// And finally  - try to paint all objects (their backbuffers) to 
				// this BackBuffer, into specified rectangle
				for(int i = 0; i < m_iObjectCount; i++)
				{
					if (m_ObjectArray[i] != null)
					{
						m_ObjectArray[i].drawInRect(g, rArea);
					}
				}
			}
			// else if transparent
			else
			{
				// Paint all sub-objects to given specified destination
				for(int i = 0; i < m_iObjectCount; i++)
				{
					if (m_ObjectArray[i] != null)
					{
						m_ObjectArray[i].drawInRect(g, rArea);
					}
				}
			}
		}
	}
	
	/**
	 * Paints specified rectangle to the given destination.
	 * Object has transparent background and his repaint goes directly to
	 * backBuffer of its parent (or parent of its parent)
	 * 
	 * @param Graphics g		- destination graphics (surface of parents' backBuffer)
	 * @param int iParentX
	 * @param int iParentY		- shift to add to coors of objects on this object to paint properly on destination
	 * @param RectangleManagerRect rAre - rectangle to be repainted in coordinates relative to this BackBuffer
	 */
	protected void paintRectFromTransparent(Graphics g, RectangleManagerRect rArea, int iParentX, int iParentY)
	{
		// First, try to bound specified rectangle to this BackBuffer
		Rectangle r = new Rectangle(0, 0, width, height);
		if (rArea.boundToRect(r))
		{
			// if not transparent
			if (m_image != null)
			{
				// Then paint background to BackBuffer (only specified rectangle)
				paintBackgroundRect(rArea);
		
				// And finally  - try to paint all objects (their backbuffers) to 
				// this BackBuffer, into specified rectangle
				for(int i = 0; i < m_iObjectCount; i++)
				{
					if (m_ObjectArray[i] != null)
					{
						m_ObjectArray[i].drawInRect(g, rArea);
					}
				}
			}
			// else if transparent
			else
			{
				// Paint all sub-objects to given specified destination
				for(int i = 0; i < m_iObjectCount; i++)
				{
					if (m_ObjectArray[i] != null)
					{
						m_ObjectArray[i].drawInRect(g, rArea, iParentX, iParentY);
					}
				}
			}
		}
	}
	
	
	/**
	 * Paints this object to destination, in specified bounding rectangle.
	 * (For BackBuffer with objects - this actually copies from backbuffer)
	 */
	protected void drawInRect(Graphics gDest, Rectangle boundRect)
	{
		if (!m_bVisible)
			return;
		
		RectangleManagerRect rObject = new RectangleManagerRect();
		rObject.setPosAndSize(x, y, width, height);
		
		// try to bound object rectangle to wanted area..
		if (rObject.boundToRect(boundRect))
		{
			// if this object is not transparent
			if (m_image != null)
			{
				// if this object has only background and no objects - paint that picture
				if ((m_iObjectCount == 0) && (m_imgBackground != null))
				{
					// Paint bgPicture to backBuffer:
					gDest.drawImage(m_imgBackground, 
							rObject.x, rObject.y, rObject.x + rObject.width, rObject.y + rObject.height,	// dest
							rObject.x - x, rObject.y - y, 
							(rObject.x - x) + rObject.width, (rObject.y - y) + rObject.height,	// source
							m_imob);
				}
				else
				{
					// Paint object to backBuffer:
					gDest.drawImage(m_image, 
							rObject.x, rObject.y, rObject.x + rObject.width, rObject.y + rObject.height,	// dest
							rObject.x - x, rObject.y - y, 
							(rObject.x - x) + rObject.width, (rObject.y - y) + rObject.height,	// source
							m_imob);
				}
			}
			// else if transparent
			else
			{
				rObject.x -= x;
				rObject.y -= y;
				paintRectFromTransparent(gDest, rObject, x, y);
			}
		}
	}
	
	/**
	 * Paints this object to destination, in specified bounding rectangle.
	 * (For BackBuffer with objects - this actually copies from backbuffer)
	 */
	protected void drawInRect(Graphics gDest, Rectangle boundRect, int iParentX, int iParentY)
	{
		if (!m_bVisible)
			return;
		
		RectangleManagerRect rObject = new RectangleManagerRect();
		rObject.setPosAndSize(x, y, width, height);
		
		// try to bound object rectangle to wanted area..
		if (rObject.boundToRect(boundRect))
		{
			// if this object is not transparent
			if (m_image != null)
			{
				// if this object has only background and no objects - paint that picture
				if ((m_iObjectCount == 0) && (m_imgBackground != null))
				{
					// Paint bgPicture to backBuffer:
					gDest.drawImage(m_imgBackground, 
							rObject.x + iParentX, rObject.y + iParentY, 
							rObject.x + iParentX + rObject.width, rObject.y + iParentY + rObject.height,	// dest
							rObject.x - x, rObject.y - y, 
							(rObject.x - x) + rObject.width, (rObject.y - y) + rObject.height,	// source
							m_imob);
				}
				else
				{
					// Paint object to backBuffer:
					gDest.drawImage(m_image, 
							rObject.x + iParentX, rObject.y + iParentY, 
							rObject.x + iParentX + rObject.width, rObject.y + iParentY + rObject.height,	// dest
							rObject.x - x, rObject.y - y, 
							(rObject.x - x) + rObject.width, (rObject.y - y) + rObject.height,	// source
							m_imob);
				}
			}
			// else if transparent
			else
			{
				RectangleManagerRect rThis = new RectangleManagerRect();
				rThis.setPosAndSize(0, 0, width, height);
			}
		}
	}
	

	/**
	 * Prepares (clears) given rectangle in backBuffer 
	 * for painting objects onto it..
	 */
	protected void paintBackgroundRect(Rectangle r)
	{
		if (m_imgBackground != null)
		{
			m_image.getGraphics().drawImage(m_imgBackground, 
							r.x, r.y, r.x + r.width, r.y + r.height,
							r.x, r.y, r.x + r.width, r.y + r.height,
							Color.white, m_imob);		
		}
		else if (m_colorBackground != null)
		{
			m_image.getGraphics().setColor(m_colorBackground);
			m_image.getGraphics().fillRect(r.x, r.y, r.width, r.height);
		}
		else
		{
			// skip painting - this object has transparent background
		}
	}

	
	/**
	 * Returns if object is visible (true), or hidden (false)
	 */
	public boolean isVisible()
	{
		return m_bVisible;
	}

	
	/**
	 * private int getPixel(int x, int y, Image img)
	 * IN:	int x		- x coordinate on parent
	 *		int y		- y coordinate on parent
	 * OUT: color of pixel
	 */
	public int getPixel(int x, int y)
	{
		if (m_iObjectCount == 0)
			return getPixel(x, y, m_imgBackground);
		else
			return getPixel(x, y, m_image);
	}
	
	
	/**
	 * private int getPixel(int x, int y, Image img)
	 * IN:	int x		- x coordinate on parent
	 *		int y		- y coordinate on parent
	 *		Image img	- source image
	 * OUT: color of pixel
	 */
	public int getPixel(int x, int y, Image img)
	{
		if ((x < this.x) || (y < this.y) ||
			(x >= this.x + this.width) || (y >= this.y + this.height))
			return -1;

		int[] pixels = new int[1 * 1];
		
		PixelGrabber pg = new PixelGrabber(img, x - this.x, y - this.y, 1, 1, pixels, 0, 1);
		try 
		{
			pg.grabPixels();	
		} 
		catch (InterruptedException e) 
		{
			System.err.println("interrupted waiting for pixels!");	    
			return -1;	
		}
		
		if ((pg.getStatus() & ImageObserver.ABORT) != 0) 
		{
			System.err.println("image fetch aborted or errored");	    
			return -1;	
		}
		
		return pixels[0];	//(pixels[0] & 0xffffff);
	}

	public boolean hasPixelAt(int x, int y)
	{
		int iPix = getPixel(x, y);
		if ( iPix == -1)
			return false;
			
		int alpha = (iPix >> 24) & 0xff;
			
		if (alpha == 0)		// 0 = transparent
			return false;
		else
			return true;
	}
	
	public void invalidate()
	{
		m_RectangleManager.addRectangle(0, 0, width, height);
	}
	
}
