/**
 * CoralApplet.java
 * v1.0
 *
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 *	
 * Description:
 *		Represents applet for playing Coral game.
 *
 *** LOG ***
 * Jul/1999
 * ~~~~~~~~
 *	created, developed gamma version
 *
 * 19.Dec.1999. (SUN)
 * ~~~~~~~~~~~~~~~~~~
 *	- continued...
 *
 * 2.Jun.2000. (SAT)
 * ~~~~~~~~~~~~~~~~~~
 *	- finished!
 */
import java.awt.*;
import java.awt.image.*;
import java.awt.image.ImageObserver;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.Cursor;

import java.util.Date;

import java.applet.*;
import java.applet.AudioClip;
import java.net.URL;

public class CoralApplet
		extends Applet
		implements Runnable, MouseListener, MouseMotionListener, KeyListener
{

	/** GAME OBJECT ********************************************************/

	CoralGame		m_game;

	
	/** SOUND CLIPS ********************************************************/

	AudioClip m_clipPlayerGun;
	AudioClip m_clipPlayerGunHat;
	AudioClip m_clipFightGun;
	AudioClip m_clipFightGunHat;

	boolean m_bUseSoundEffects = true;

	
	/** LOADING ************************************************************/

	MediaTracker tracker;			// used for loading images

	String		m_strLoadingMessage = new String("Loading data...");
	BackBuffer	m_bbLoading;		// when first image will be loaded - it wil be used as background for loading text
	BackBuffer	m_bbPercentage;		// small sprite on m_bbLoading background with loading message
	int			m_iPercentage;		// current percentage of loaded images..

	boolean m_bInitialized = false;		// true when all loaded and game initialized

	
	/** APPLET & THREAD HANDLERS *******************************************/

	public void init()
	{
		this.setBackground(Color.black);
		this.setForeground(Color.white);

		addMouseListener(this);
		addMouseMotionListener(this);
		addKeyListener(this);

		// set mouse cursor
		setCursor(new Cursor(Cursor.CROSSHAIR_CURSOR));
	}


	public void destroy()
	{
		if (m_threadGame != null && m_threadGame.isAlive())
		{
			m_threadGame.stop();
			m_threadGame = null;
		}
	}

	/** THREADING **********************************************************/

	Thread m_threadGame;		// main running thread


	/**
	 * start() handler
	 */
	public void start()
	{
		// create running thread
		if (m_threadGame == null)
		{
			m_threadGame = new Thread(this);
		}
		m_threadGame.start();
	}

	/**
	 * run() handler
	 */
	public void run()
	{
		loadImagesAndSounds();
		m_game.init();
		repaint();

		m_bInitialized = true;

		//
		// Get "first time"
		//
		long lNewTime;
		long lLastTime = System.currentTimeMillis();

		//
		// game-loop
		//
		while(true)
		{
			lNewTime = System.currentTimeMillis();

			// do action
			m_game.action(lNewTime - lLastTime);

			// repaint
			update(getGraphics());

			updateMouseCursor();
			doSoundEffects();

			lLastTime = lNewTime;

			// give little time to system...
			try
			{
				m_threadGame.sleep(10);
			}
			catch(InterruptedException e) {}
		}
	}

	/**
	 * stop() handler
	 */
	public void stop()
	{
		if (m_threadGame != null && m_threadGame.isAlive())
		{
			m_threadGame.stop();
			m_threadGame = null;
		}
	}

	/** UTILITY ************************************************************/

	/**
	 * Loads images and sounds
	 * @return true if ok, false when failed loading..
	 */
	private boolean loadImagesAndSounds()
	{
		m_strLoadingMessage = "Loading sounds...";
		repaint();

		try
		{
			m_clipPlayerGun = getAudioClip(getDocumentBase(), "snd/player_gun.au");
			m_clipPlayerGunHat = getAudioClip(getDocumentBase(), "snd/player_gun_hat.au");
			m_clipFightGun = getAudioClip(getDocumentBase(), "snd/fight_gun.au");
			m_clipFightGunHat = getAudioClip(getDocumentBase(), "snd/fight_gun_hat.au");
		}
		catch(Exception e)
		{
			m_strLoadingMessage = "Failed loading sounds...";
			repaint();
			try
			{
				m_threadGame.sleep(1000);
			}
			catch(InterruptedException oops) {}
		}

		m_strLoadingMessage = "Loading images...";
		repaint();

		int iTrackerImgCount = 0;
		tracker = new MediaTracker(this);

		m_game = new CoralGame(this, this);

		// load images for m_game
		for ( int i=0; i<m_game.NUMBER_OF_IMAGES; i++ )
		{
			m_game.m_images[i] = getImage(getDocumentBase(), "gfx/" + m_game.imgName[i]);
			tracker.addImage(m_game.m_images[i], iTrackerImgCount);
			iTrackerImgCount++;
		}

		//
		// Start downloading images, and wait until they're loaded
		// (before requesting repaints)
		//

		for (int iCurrent = 0; iCurrent <= iTrackerImgCount; iCurrent++)
		{
			try
			{
				tracker.waitForID(iCurrent);
			}
			catch(InterruptedException e)
			{
				m_strLoadingMessage = "Could not load image " + String.valueOf(iCurrent);
				repaint();
				return false;
			}

			// first image is Fight background, and when it's loaded - we'll use it as loading background...
			if (iCurrent == m_game.IMG_BACKGROUND_FIGHT)
			{
				m_bbLoading = new BackBuffer();
				m_bbLoading.createForImage(this, this);
				m_bbLoading.setImage(m_game.m_images[0]);

				Image img = createImage(400, 20);

				m_bbPercentage = new BackBuffer();
				m_bbPercentage.createForImage(this, this);
				m_bbPercentage.setImage(img);
				m_bbPercentage.moveObject((640 - m_bbPercentage.width)/2, (400 - m_bbPercentage.height)/2);
				m_bbLoading.addObject(m_bbPercentage);
			}

			m_iPercentage = iCurrent * 100 / iTrackerImgCount;
			if (m_iPercentage > 100)
				m_iPercentage = 100;

			m_strLoadingMessage = "Loading (" + String.valueOf(m_iPercentage) + " %)";
			repaint();
		}

		m_iPercentage = 100;
		m_strLoadingMessage = "Initializing...";
		repaint();

		// and one more check..
		try
		{
			tracker.waitForAll();
		}
		catch(InterruptedException e)
		{
			m_strLoadingMessage = "Could not load images...";
			return false;
		}

		if (!tracker.checkAll())
		{
            // error - images should have been loaded by now...
			m_strLoadingMessage = "Could not load all images...";
			repaint();
			return false;
		}
		else
		{
			m_strLoadingMessage = "Images loaded - initializing...";
			repaint();
			return true;
		}
	}

	int m_iMouseCursor = -1;

	public void updateMouseCursor()
	{
		if (m_iMouseCursor != m_game.m_iMouseCursor)
		{
			m_iMouseCursor = m_game.m_iMouseCursor;
			if (m_iMouseCursor == m_game.MOUSE_CURSOR_ARROW)
				setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
			else if (m_iMouseCursor == m_game.MOUSE_CURSOR_HAND)
				setCursor(new Cursor(Cursor.HAND_CURSOR));
			else if (m_iMouseCursor == m_game.MOUSE_CURSOR_CROSS)
				setCursor(new Cursor(Cursor.CROSSHAIR_CURSOR));
		}
	}

	public void doSoundEffects()
	{
		if (!m_bUseSoundEffects)
		{
			m_game.m_bSndPlayerGun = false;
			m_game.m_bSndPlayerGunHat = false;
			m_game.m_bSndFightGun = false;
			m_game.m_bSndFightGunHat = false;
			return;
		}

		if (m_game.m_bSndPlayerGun)
		{
			m_game.m_bSndPlayerGun = false;
			m_clipPlayerGun.play();
		}
		else if (m_game.m_bSndPlayerGunHat)
		{
			m_game.m_bSndPlayerGunHat = false;
			m_clipPlayerGunHat.play();
		}
		else if (m_game.m_bSndFightGun)
		{
			m_game.m_bSndFightGun = false;
			m_clipFightGun.play();
		}
		else if (m_game.m_bSndFightGunHat)
		{
			m_game.m_bSndFightGunHat = false;
			m_clipFightGunHat.play();
		}
	}


	/** GRAPHIC HANDLERS ******************************************************/

	public void update(Graphics g)
	{
		if (!m_bInitialized)
		{
			paintLoading(g);
		}
		else
		{
			m_game.update(g, 0, 0);
		}
	}

	public void paint(Graphics g)
	{
		if (!m_bInitialized)
		{
			paintLoading(g);
		}
		else
		{
			m_game.paint(g, 0, 0);
		}
	}

	public void paintLoading(Graphics g)
	{
		if (m_bbLoading == null)
		{
			Dimension d = getSize();
			g.setColor(Color.black);
			g.fillRect(0, 0, d.width, d.height);
			g.setColor(Color.white);
			g.drawString(m_strLoadingMessage, 5, 15);
			return;
		}
		else
		{
			Graphics gg = m_bbPercentage.m_imgBackground.getGraphics();

			gg.setColor(Color.black);
			gg.fillRect(0, 0, m_bbPercentage.width, m_bbPercentage.height);
			gg.setColor(Color.red);
			gg.fillRect(3, 3, (m_bbPercentage.width - 6)* m_iPercentage / 100, m_bbPercentage.height - 6);

			gg.setColor(Color.white);
			gg.drawString(m_strLoadingMessage, 6, 14);
			m_bbPercentage.invalidate();

			m_bbLoading.paint(g, 0, 0);
		}
	}

	/** MOUSE HANDLERS ********************************************************/

	public void mouseClicked( MouseEvent e )
	{
		//
	}

	public void mouseMoved(MouseEvent e)
	{
		if (m_bInitialized)
			m_game.onMouseMoved(e.getX(), e.getY());
	}

	public void mouseDragged(MouseEvent e)
	{
		if (m_bInitialized)
			m_game.onMouseMoved(e.getX(), e.getY());
	}

	public void mouseEntered(MouseEvent e)
	{
		if (m_bInitialized)
			m_game.onMouseMoved(e.getX(), e.getY());
	}

	public void mouseExited(MouseEvent e)
	{
		if (m_bInitialized)
			m_game.onMouseMoved(e.getX(), e.getY());
	}

	public void mousePressed(MouseEvent e)
	{
		if (m_bInitialized)
			m_game.onMouseClicked(e.getX(), e.getY());
	}

	public void mouseReleased(MouseEvent e)
	{
		//
	}

	/** KEYBOARD HANDLERS **************************************************/

	boolean __KEYBOARD_HANDLERS_______;

	public void keyPressed(KeyEvent e)
	{
		if (m_bInitialized)
		{
			int key = e.getKeyCode();
			if (key == e.VK_ESCAPE)
			{
				m_game.onKeyEsc();
			}
			else if (key == e.VK_S)
			{
				m_bUseSoundEffects = !m_bUseSoundEffects;
			}
		}
	}

	public void keyReleased(KeyEvent e)
	{
		//
	}

	public void keyTyped(KeyEvent e)
	{
		//
	}
}
