/**
 * ShootArea.java 
 * v1.0
 * 
 * Project: Coral game
 *
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 *
 *** DESCRIPTION & USAGE ***
 *		
 * This class represents "shooting area".
 * That is rectangle sprite on some backBuffer that has its door,
 * and a guys to show behind opening door.
 * 
 * To use it, do following:
 * 
 * 1. Create it:
 *	
 *		ShootArea sa = new ShootArea(yourDoor, 5, m_component, m_imob);
 * 
 * 2. Initialize it:
 * 
 * 		sa.moveObject(100, 200);
 * 		parent.addObject(sa);
 * 		sa.addBadGuy(bad_guy, 100);
 * 		sa.addGoodGuy(good_guy, 100);
 * 
 * 3. Activate it:
 * 
 * 		sa.setStateSleeping();
 * 
 * 4. Now, parent and its ActiveBackBuffer logic will run action in
 *    ShootArea.. You should call onMouseClick(x, y) to provide player
 *	  input.
 *    First, area is in STATE_SLEEPING. Then, one of given guys is choosen,
 *    and area comes in STATE_CYCLE.
 *	  It will turn back to STATE_SLEEPING when good guy is dead, or requests
 *    to do door-closing, or when bad guy is shot, or did shoot player.
 *		
 *	  Check RESPONSE_* to see possible responses from this ShootArea.
 *
 *** LOG ***
 * 08.Aug.1999.
 * ~~~~~~~~~~~~
 *	10:27..	- created
 * 
 * 22.Dec.1999.
 * ~~~~~~~~~~~~
 *	- trying to figure out new concept..
 * 
 * 23.Dec.1999.
 * ~~~~~~~~~~~~
 *	10:27..	- got new concept!
 *
 * 30.Dec.1999.
 * ~~~~~~~~~~~~
 *	12:21..	- v1.0 finished!
 */
import java.awt.*;
import java.awt.image.ImageObserver;
import java.lang.Math;
import java.util.Vector;

public class ShootArea
	extends ActiveBackBuffer
{
	/** PARENT DATA ********************************************************/

	ImageObserver		m_imob;
	Component			m_component;

	/** STATES *************************************************************/
	
	final static int STATE_SLEEPING				= 1;	// Door is closed, no guy active (door did setActionWhenTime(..) to get to STATE_CYCLE)
	final static int STATE_CYCLE				= 2;	// Door are opening, guy does action, door will close and finally get back to STATE_SLEEPING
		
	int m_iState = STATE_SLEEPING;	// Door is closed, no guy active..
	
	/** RESPONSES **********************************************************/
	
	final static int RESPONSE_NONE					= 1;
	final static int RESPONSE_BAD_GUY_SHOOT_PLAYER	= 2;
	final static int RESPONSE_PLAYER_SHOOT_BAD_GUY	= 3;
	final static int RESPONSE_PLAYER_SHOOT_GOOD_GUY	= 4;
	final static int RESPONSE_PLAYER_SHOOT_HAT		= 5;
	
	public int m_iResponse = RESPONSE_NONE;
	
	
	/**
	 * Shoot area has its door and guys.
	 * There may be >1 possible guys to show up in area... (sequentially)
	 */
	
	Door m_door;					// given door object

	Vector m_vectorBadGuys = new Vector(3, 3);
	Vector m_vectorBadGuysPossibilities = new Vector(3, 3);
	
	Vector m_vectorGoodGuys = new Vector(3, 3);
	Vector m_vectorGoodGuysPossibilities = new Vector(3, 3);
	
	private BadGuy	m_currentBadGuy;
	private GoodGuy	m_currentGoodGuy;
	

	/** CODE ***************************************************************/
	
	/**
	 * Construction
	 */
	public ShootArea(Door d, int iPossibleGuys, Component c, ImageObserver imob)
	{
		// remember parent data
		m_component = c;
		m_imob = imob;
		
		// create BackBuffer
		createTransparent(d.width, d.height, m_component, m_imob);
		m_strName = "SHOOTAREA";
		
		// remember door
		m_door = d;
		addObject(d);		
	}
	
	
	/**
	 * Adds bad guy as another possible guy to show in area
	 */
	public void addBadGuy(BadGuy guy, int iPossibility)
	{
		m_vectorBadGuys.addElement(guy);
		m_vectorBadGuysPossibilities.addElement(new Integer(iPossibility));
	}

	/**
	 * Adds good guy as another possible guy to show in area
	 */
	public void addGoodGuy(GoodGuy guy, int iPossibility)
	{
		m_vectorGoodGuys.addElement(guy);
		m_vectorGoodGuysPossibilities.addElement(new Integer(iPossibility));
	}

	/** ACTION *************************************************************/
	boolean __ACTION____________________;
		
	public void setStateSleeping()
	{
		m_iState = STATE_SLEEPING;
		m_iResponse = RESPONSE_NONE;
		setActionWhenTime(getNextOpeningDelay());
		m_door.setStateClosed();
	}
	
	private void setStateCycle()
	{
		m_iState = STATE_CYCLE;
		m_door.setStateCanOpen();
		chooseAndActivateNewGuy();
		
		// we are going to check child RESPONSE_* all the time..
		setActionContinuous();
	}

	
	public void doNextAction()
	{
		if (m_iState == STATE_SLEEPING)
		{
			setStateCycle();
		}
		else
		{
			setStateSleeping();	// ERROR, this should not happen
		}
	}
	
	
	public long doContinuousAction(long lTimeForAction)
	{
		if (m_iState == STATE_CYCLE)
		{
			// Check new possible state
				
			if (m_currentBadGuy != null)
			{
				if ((m_currentBadGuy.m_iResponse == m_currentBadGuy.RESPONSE_GUY_IS_DEAD) && 
					(m_door.m_iState == m_door.STATE_OPENED))
				{
					m_door.setStateCanClose();
				}
				else if (m_currentBadGuy.m_iResponse == m_currentBadGuy.RESPONSE_GUY_SHOOT)
				{
					m_iResponse = RESPONSE_BAD_GUY_SHOOT_PLAYER;
				}
				
				m_currentBadGuy.m_iResponse = m_currentBadGuy.RESPONSE_NONE;
				
				// check onMouseClick() for guy's responses to player's click..
			}
			
			if (m_currentGoodGuy != null)
			{
				if ((m_currentGoodGuy.m_iResponse == m_currentGoodGuy.RESPONSE_GUY_IS_DEAD) && 
					(m_door.m_iState == m_door.STATE_OPENED))
				{
					m_door.setStateCanClose();
				}
				else if ((m_currentGoodGuy.m_iResponse == m_currentGoodGuy.RESPONSE_CLOSE_DOOR) && 
					(m_door.m_iState == m_door.STATE_OPENED))
				{
					m_door.setStateCanClose();
				}
				
				m_currentGoodGuy.m_iResponse = m_currentGoodGuy.RESPONSE_NONE;
				
				// check onMouseClick() for guy's responses to player's click..
			}
			
			if (m_door.m_iState == m_door.STATE_CLOSED)
			{
				removeGuys();
				setStateSleeping();
			}
		}
		else
		{
			setStateCycle();	// ERROR, this should not happen
		}
		
		return 0;
	}

	
	/** UTILITY PROCS ******************************************************/
	
	boolean __UTILITY___________________;
	
	/**
	 * private void chooseNewGuy()
	 */
	private void chooseAndActivateNewGuy()
	{
		int iBadGuys = m_vectorBadGuys.size();
		int iGoodGuys = m_vectorGoodGuys.size();
		
		// if no possible guys - do nothing
		if ((iBadGuys == 0) && (iGoodGuys == 0))
			return;

		//
		// choose guy to use
		//

		// make soom of possibilities..
		int iSum = 0;
		int i;
		for (i=0; i<iBadGuys; i++)
		{
			iSum += ((Integer)m_vectorBadGuysPossibilities.elementAt(i)).intValue();
		}
		for (i=0; i<iGoodGuys; i++)
		{
			iSum += ((Integer)m_vectorGoodGuysPossibilities.elementAt(i)).intValue();
		}
		
		// make random number...
		Double d = new Double(Math.random() * iSum);
		int iPos = d.intValue();
		if (iPos >= iSum)
			iPos = iSum - 1;
		
		// check if we have bad guy..
		for (i=0; i<iBadGuys; i++)
		{
			int iPossibility = ((Integer)m_vectorBadGuysPossibilities.elementAt(i)).intValue();
			if (iPos <= iPossibility)
			{
				// this is the guy..
				m_currentBadGuy = (BadGuy) m_vectorBadGuys.elementAt(i);
				m_currentBadGuy.activate(this);
				return;
			}
			iPos -= iPossibility;
		}
		
		// check if we have good guy..
		for (i=0; i<iGoodGuys; i++)
		{
			int iPossibility = ((Integer)m_vectorGoodGuysPossibilities.elementAt(i)).intValue();
			if (iPos <= iPossibility)
			{
				// this is the guy..
				m_currentGoodGuy = (GoodGuy) m_vectorGoodGuys.elementAt(i);
				m_currentGoodGuy.activate(this);
				return;
			}
			iPos -= iPossibility;
		}
	}
	
	

	/**
	 * Processes mouse click (in case it shoot the guy or his hat).
	 * @param int x		- x coord. relative to parent of this object
	 * @param int y		- y coord. relative to parent this object
	 */
	public void onMouseClick(int x, int y)
	{
		// if mouse is outside this area - do nothing
		if ((x < this.x) || (y < this.y) ||
			(x >= this.x + this.width) || (y >= this.y + this.height))
			return;

		// translate coordinates into this object
		x = x - this.x;
		y = y - this.y;
		
		// if shoot door - just return
		if (m_door.hasPixelAt(x, y))
			return;

		// check bad guy..
		if (m_currentBadGuy != null)
		{
			m_currentBadGuy.onMouseClick(x, y);
			if (m_currentBadGuy.m_iResponse == m_currentBadGuy.RESPONSE_HAT_IS_SHOT)
			{
				m_iResponse = RESPONSE_PLAYER_SHOOT_HAT;
				m_currentBadGuy.m_iResponse = m_currentBadGuy.RESPONSE_NONE;
			}
			else if ((m_currentBadGuy.m_iResponse == m_currentBadGuy.RESPONSE_GUY_IS_SHOT) ||
					 (m_currentBadGuy.m_iResponse == m_currentBadGuy.RESPONSE_GUY_IS_SHOT_TOO_EARLY))
			{
				m_iResponse = RESPONSE_PLAYER_SHOOT_BAD_GUY;
				m_currentBadGuy.m_iResponse = m_currentBadGuy.RESPONSE_NONE;
			}
		}
		// check good guy..
		else if (m_currentGoodGuy != null)
		{
			m_currentGoodGuy.onMouseClick(x, y);
			
			if (m_currentGoodGuy.m_iResponse == m_currentGoodGuy.RESPONSE_GUY_IS_SHOT)
			{
				m_iResponse = RESPONSE_PLAYER_SHOOT_GOOD_GUY;
				m_currentGoodGuy.m_iResponse = m_currentGoodGuy.RESPONSE_NONE;
			}
			else if (m_currentGoodGuy.m_iResponse == m_currentGoodGuy.RESPONSE_HAT_IS_SHOT)
			{
				m_iResponse = RESPONSE_PLAYER_SHOOT_HAT;
				m_currentGoodGuy.m_iResponse = m_currentGoodGuy.RESPONSE_NONE;
			}
		}
	}
	
	/** CALCULATION OF NEXT OPENING DELAY **********************************/

	long m_lMinOpeningDelay = 1000;	// min. delay (in milliseconds) to wait to open area again
	long m_lMaxOpeningDelay = 5000;	// max. delay (in milliseconds) to wait to open area again

	private long getNextOpeningDelay()
	{
		double d = m_lMinOpeningDelay + (Math.random() * (m_lMaxOpeningDelay - m_lMinOpeningDelay));
		Double dd = new Double(d);
		return dd.longValue();
	}
	

	/**
	 * Removes current bad and good guys from door...
	 */
	public void removeGuys()
	{
		if (m_currentBadGuy != null)
		{
			m_currentBadGuy.dismiss(this);
			m_currentBadGuy = null;
		}
						
		if (m_currentGoodGuy != null)
		{
			m_currentGoodGuy.dismiss(this);
			m_currentGoodGuy = null;
		}
	}

	/**
	 * Removes all possible guys..
	 */
	public void removeAllGuys()
	{
		removeGuys();
		m_vectorBadGuys.removeAllElements();
		m_vectorBadGuysPossibilities.removeAllElements();
	
		m_vectorGoodGuys.removeAllElements();
		m_vectorGoodGuysPossibilities.removeAllElements();
	}
}
